////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>
NS_ASSUME_NONNULL_BEGIN

@protocol UserInformationProtocol;
@class OrchestrationUser;

/**
 * The {@link UserManagerProtocol} allows to handle user management related features.
 */

NS_REFINED_FOR_SWIFT
@protocol UserManagerProtocol <NSObject>

/**
 * Check whether the provided user identifier corresponds to an activated user.
 * @param orchestrationUser the user to check
 * @param error a NSError raised if there is an error during the process.
 * @return true if the user is activated, If an error occurred the reason will be put in the error.
 */
- (BOOL)isUserActivated:(OrchestrationUser *)orchestrationUser error:(NSError **)error NS_SWIFT_NOTHROW;

/**
 * Retrieves the information related to a specific user.
 * @param orchestrationUser the user
 * @param error a NSError raised if there is an error during the process.
 * @return nil if an error occured, or a {@link UserInformationProtocol} object
 */
- (id<UserInformationProtocol> _Nullable)userInformationForOrchestrationUser:(OrchestrationUser *)orchestrationUser error:(NSError **)error;

/**
 * Deactivates a specified user.
 * @param error a NSError raised if there is an error during the process.
 * @param orchestrationUser the user to deactivate
 * @return true if the call is successful, If an error occurred the reason will be put in the error.
 */
- (BOOL)deleteUserWithOrchestrationUser:(OrchestrationUser *)orchestrationUser error:(NSError **)error;

/**
 * Deactivates all the user.
 * @param error a NSError raised if there is an error during the process.
 * @return true if the call is successful, If an error occurred the reason will be put in the error.
 */
- (BOOL)deleteAllOrchestrationUsersWithError:(NSError **)error;

/**
 * Retrieves the list of activated users.
 * @return a list of {@link OrchestrationUser}
 */
- (NSArray<OrchestrationUser *> *)users __deprecated_msg("use [UserManagerProtocol usersWithError:] to have the error");

/**
 * Retrieves the list of activated users.
 * @param error a NSError raised if there is an error during the process.
 * @return a list of {@link OrchestrationUser}
 */
- (NSArray<OrchestrationUser *> *_Nullable)usersWithError:(NSError **)error;

@end

NS_ASSUME_NONNULL_END
